/*
 * The MIT License (MIT)
 * <p/>
 * Copyright (c) 2017
 * <p/>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p/>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p/>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.techcode.scrawler;

import com.google.common.base.Splitter;
import edu.stanford.nlp.tagger.maxent.MaxentTagger;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.jsoup.Connection;
import org.jsoup.Jsoup;

import java.io.IOException;
import java.text.Normalizer;
import java.text.ParseException;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;

/**
 * Abstract crawl implementation.
 */
@RequiredArgsConstructor
public abstract class AbstractCrawler {

    // Some constant pattern
    private static final Pattern NONLATIN = Pattern.compile("[^\\w-]");
    private static final Pattern WHITESPACE = Pattern.compile("[\\s]");

    // Joiner
    static final Splitter SPLITTER = Splitter.on(" ").omitEmptyStrings();

    // NPL Tagger
    static MaxentTagger TAGGER = new MaxentTagger(MaxentTagger.DEFAULT_JAR_PATH);

    // Rules to apply on every crawl
    static Map<String, Pattern> RULES = new TreeMap<String, Pattern>() {{
        put("word", Pattern.compile("([a-zA-Z]+)_nn", Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE));
    }};

    @Getter private final String baseUri;

    @Getter @Setter private String uri;

    @Getter private final Map<String, String> result;

    /**
     * Gets the url to crawl.
     *
     * @return url to crawl.
     */
    public String getUrl() {
        return uri != null ? baseUri + uri : baseUri;
    }

    /**
     * Gets a connection to crawl.
     *
     * @return connection to crawl.
     */
    public Connection getConnection() {
        return Jsoup.connect(getUrl()).followRedirects(false);
    }

    /**
     * Crawl a webpage based on uri entrypoint.
     */
    public abstract void crawl() throws IOException, ParseException;

    /**
     * Normalize raw data.
     *
     * @param data raw data to normalize
     * @return data normalized.
     */
    static String normalize(String data) {
        String nowhitespace = WHITESPACE.matcher(data).replaceAll("-");
        String normalized = Normalizer.normalize(nowhitespace, Normalizer.Form.NFD);
        String slug = NONLATIN.matcher(normalized).replaceAll("");
        return slug.toLowerCase(Locale.ENGLISH).replaceAll("-", " ");
    }

}
