/*
 * The MIT License (MIT)
 * <p/>
 * Copyright (c) 2017
 * <p/>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p/>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p/>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.techcode.scrawler;

import com.google.common.base.Strings;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

import java.io.IOException;
import java.text.ParseException;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Abstract crawl implementation.
 */
public class TopicCrawler extends AbstractCrawler {

    // Gson instance
    private static final Gson GSON = new GsonBuilder().create();

    // Date parser
    private static final DateTimeFormatter DATE_PARSER = DateTimeFormat
            .forPattern("MMM d, yyyy 'at' h:mm a")
            .withLocale(Locale.ENGLISH)
            .withDefaultYear(2016);

    // ISO formatter
    private static final DateTimeFormatter ISO_FORMATTER = ISODateTimeFormat.dateTime().withLocale(Locale.ENGLISH);

    TopicCrawler(String baseUri, Map<String, String> result) {
        super(baseUri, result);
    }

    @Override public void crawl() throws IOException, ParseException {
        // Attempt to crawl data
        Document doc = getConnection().get();
        getResult().put("title", normalize(doc.body().select(".titleBar h1").text()));

        for (Element el : doc.body().select(".messageList > li")) {
            getResult().put("content", normalize(el.select(".messageContent").text()));
            getResult().put("author", el.attr("data-author"));
            String date = el.select(".datePermalink .DateTime").attr("title");
            if (Strings.isNullOrEmpty(date)) continue;
            getResult().put("@timestamp", ISO_FORMATTER.print(DATE_PARSER.parseDateTime(date)));

            // Iterate over each entry
            StringBuilder builder = new StringBuilder();
            String content = getResult().get("content");

            // Tag data
            String taggedData = TAGGER.tagString(content);
            for (String word : SPLITTER.split(taggedData)) {
                // Iterate over each RULES
                for (Map.Entry<String, Pattern> rule : RULES.entrySet()) {
                    Matcher matcher = rule.getValue().matcher(word);
                    if (matcher.find()) {
                        String match = matcher.group(1);
                        builder.append(match).append(" ");
                    }
                }
            }

            // Extract content
            builder.setLength(Math.max(builder.length() - 1, 0));
            getResult().put("content_nlp", builder.toString());
            builder.setLength(0);

            // Add some informations
            getResult().put("uri", getUrl());
            System.out.println(GSON.toJson(getResult()));
        }

        Element nextPage = doc.body().select(".PageNav nav > a:last-child").first();
        if (nextPage != null && nextPage.text().startsWith("Next")) {
            setUri(nextPage.attr("href"));
            crawl();
        }
    }

}